from fastapi import FastAPI, HTTPException, Request
import httpx, os, asyncio
from config import settings
import redis
import time
import json

app = FastAPI()
r = redis.from_url(settings.CACHE_REDIS_URL, decode_responses=True)

OPENAI_HEADERS = {
    "Authorization": f"Bearer {settings.OPENAI_API_KEY}",
    "Content-Type": "application/json"
}

RATE_LIMIT_KEY = "rl:{}"

async def call_openai(payload: dict):
    url = f"{settings.OPENAI_API_BASE}/responses"
    async with httpx.AsyncClient(timeout=60) as client:
        resp = await client.post(url, headers=OPENAI_HEADERS, json=payload)
        resp.raise_for_status()
        return resp.json()

@app.post("/v1/chat")
async def chat(req: Request):
    body = await req.json()
    key = RATE_LIMIT_KEY.format(req.client.host)
    now = int(time.time())
    window = 60
    count = r.get(key)
    if count and int(count) >= settings.RATE_LIMIT_PER_MIN:
        raise HTTPException(status_code=429, detail="rate limit exceeded")
    else:
        r.incr(key, 1)
        r.expire(key, window)
    import hashlib
    h = hashlib.sha256(json.dumps(body, sort_keys=True).encode()).hexdigest()
    cached = r.get(f"cache:{h}")
    if cached:
        return {"cached": True, "response": json.loads(cached)}
    payload = {
        "model": settings.MODEL,
        "input": body.get("input")
    }
    try:
        resp = await call_openai(payload)
    except httpx.HTTPStatusError as e:
        raise HTTPException(status_code=e.response.status_code, detail=str(e))
    out = resp
    r.setex(f"cache:{h}", 60*5, json.dumps(out))
    return out

@app.get("/health")
def health():
    return {"status": "ok"}
