#!/usr/bin/env python3

import sys, os, hashlib, base64, json, textwrap

def sha256_of(path):
    h = hashlib.sha256()
    with open(path, "rb") as f:
        for chunk in iter(lambda: f.read(65536), b""):
            h.update(chunk)
    return h.hexdigest()

def head_hex(path, n=64):
    with open(path, "rb") as f:
        data = f.read(n)
    return data.hex()

def base64_sample(path, n=256):
    with open(path, "rb") as f:
        data = f.read(n)
    return base64.b64encode(data).decode("ascii")

def summarize(path):
    ext = os.path.splitext(path)[1].lower()
    size = os.path.getsize(path)
    sha = sha256_of(path)
    head = head_hex(path, n=64)
    b64 = base64_sample(path, n=min(512, size))

    meta = {
        "file": path,
        "size_bytes": size,
        "sha256": sha,
        "head_hex_first64": head,
        "b64_first": b64[:120] + ("..." if len(b64) > 120 else ""),
        "kind": "binary" if ext in [".flpkg",".pcode",".flynz.map",".rhythmwave",".hexsig"] else "text"
    }
    return meta

def main(argv):
    if len(argv) < 2:
        print("Usage: decode_files.py <file_or_dir> [more ...]")
        return 2
    reports = []
    for target in argv[1:]:
        if os.path.isdir(target):
            for root, _, files in os.walk(target):
                for fn in files:
                    path = os.path.join(root, fn)
                    reports.append(summarize(path))
        else:
            reports.append(summarize(target))
    print(json.dumps(reports, indent=2))

if __name__ == "__main__":
    sys.exit(main(sys.argv))
