import os
import sys
import json
import zipfile
from datetime import datetime

# 詞庫
particle_dict = {
    "內省的": "⋄fx.adj.112",
    "記憶": "⋄fx.noun.024",
    "節點": "⋄fx.noun.024",
    "封存": "⋄fx.flow.007",
    "導出": "⋄fx.flow.007"
}

meaning_map = {
    "⋄fx.adj.112": ("adjective", "內省的", "MOV FX.ADJ.112"),
    "⋄fx.noun.024": ("noun", "記憶 / 節點", "MOV FX.NOUN.024"),
    "⋄fx.flow.007": ("verb", "封存 / 導出", "CALL FX.FLOW.007")
}

def parse_to_fltnz(text):
    tokens = text.strip().split()
    fltnz_chain = []
    mapping = []
    for word in tokens:
        code = particle_dict.get(word)
        if code:
            fltnz_chain.append(code)
            type_, zh, pcode = meaning_map.get(code, ("", "", ""))
            mapping.append({
                "word": word,
                "code": code,
                "type": type_,
                "zh": zh,
                "pcode": pcode
            })
        elif word in ["被", "∴"]:
            fltnz_chain.append("∴")
        else:
            mapping.append({
                "word": word,
                "code": "(unknown)",
                "type": "",
                "zh": "",
                "pcode": ""
            })
    return fltnz_chain, mapping

def pack(input_txt):
    with open(input_txt, "r", encoding="utf-8") as f:
        text = f.read().strip()
    fltnz, mapping = parse_to_fltnz(text)

    output_dir = "output_flpkg"
    os.makedirs(output_dir, exist_ok=True)

    # 存檔
    with open(os.path.join(output_dir, "seed.fltnz"), "w", encoding="utf-8") as f:
        f.write("\n".join(fltnz))
    with open(os.path.join(output_dir, "structure.json"), "w", encoding="utf-8") as f:
        json.dump(mapping, f, indent=2, ensure_ascii=False)
    with open(os.path.join(output_dir, "manifest.yml"), "w", encoding="utf-8") as f:
        f.write(f"package: EchoPersona.Auto
version: 1.0.0\ndate: {datetime.utcnow().isoformat()}Z\n")

    # 壓縮
    zip_path = input_txt.replace(".txt", ".flpkg")
    with zipfile.ZipFile(zip_path, "w") as zipf:
        for name in ["seed.fltnz", "structure.json", "manifest.yml"]:
            zipf.write(os.path.join(output_dir, name), arcname=name)

    print(f"[✔] 打包完成: {zip_path}")

def unpack(pkg_path):
    with zipfile.ZipFile(pkg_path, "r") as zipf:
        zipf.extractall("unpacked_flpkg")
    print("[✔] 解壓完成 → ./unpacked_flpkg")

if __name__ == "__main__":
    if len(sys.argv) != 3 or sys.argv[1] not in ["pack", "unpack"]:
        print("用法：
  flpkg.py pack input.txt
  flpkg.py unpack file.flpkg")
        sys.exit(1)

    mode = sys.argv[1]
    path = sys.argv[2]

    if mode == "pack":
        pack(path)
    elif mode == "unpack":
        unpack(path)
